define(['angular', 'app'], function (angular, app) {
	"use strict";
	app.service('authenticationService',
		function ($http, $q, $window, $location, $rootScope, resourceDirectory, $injector, pageService, localResourceDirectoryService) {
			var _auth = {};
			var isAuthenticated = false;
			var loginUrl, logoutUrl, oauthInfoUrl, tokenUrl;

			var clientRedirectUri = pageService.appUri;
			var appName = (/.*\/(.*)\/$/).exec(clientRedirectUri)[1];

			_auth.readLocalResourceDirectory = function() {
				var deferred = $q.defer();
				localResourceDirectoryService.fetch().then(function(directory) {
					loginUrl = directory['login'];
					logoutUrl = directory['logout'];
					oauthInfoUrl = directory['oauth-info'];
					tokenUrl = directory['oauth-token'];
					deferred.resolve();
				});
				return deferred.promise;
			};

			_auth.authenticate = function () {
				_auth.readLocalResourceDirectory().then(function() {
					$http.get(oauthInfoUrl).then(function(response) {
						var authorizeParams = [
							"response_type=code",
							"state=" + appName + '-' + new Date().getTime(),
							"client_id=" + response.data['clientId'],
							"redirect_uri=" + clientRedirectUri,
							"scope=read"
						];

						var loginFormUrl = loginUrl + "?" + authorizeParams.join("&");
						$window.open(loginFormUrl, "_self");
					});
				});
			};

			_auth.checkForAuthCode = function(queryString) {
				var params = {};
				var regex = /([^&=]+)=([^&]*)/g;
				var m;

				while (m = regex.exec(queryString)) {
					params[decodeURIComponent(m[1])] = decodeURIComponent(m[2]);
				}

				var sessionCode  = sessionStorage.getItem('code');
				var sessionState = sessionStorage.getItem('state');

				if(params['code'] && (params['code'] !== sessionCode)
					&& params['state'] && (params['state'] !== sessionState)) {
					_auth.saveSessionStates(params['code'], params['state']);
				}
				return {code: params['code'], state: params['state']};
			};

			_auth.saveSessionStates = function(code, state) {
				sessionStorage.setItem('code',  code);
				sessionStorage.setItem('state', state);
			};

			_auth.getNewToken = function (code) {
				var deferred = $q.defer();

				_auth.readLocalResourceDirectory().then(function () {
					$http.get(oauthInfoUrl).then(function (oauthInfoResponse) {
						var tokenUrlWithRedirect = tokenUrl + '?code=' + code + '&redirect_uri=' + clientRedirectUri;
						$http.get(tokenUrlWithRedirect).then(function (response) {
							var token = response.data['access_token'];
							_auth.storeSessionToken(token);
							sessionStorage.removeItem('code');
							sessionStorage.removeItem('state');
							deferred.resolve(clientRedirectUri);
						}, function() {
							$window.open(clientRedirectUri, "_self");
						});
					});
				});

				return deferred.promise;
			};

			_auth.storeSessionToken = function(token) {
				if (token !== undefined && token !== 'undefined' && token !== null && token !== 'null') {
					sessionStorage.setItem('token', JSON.stringify(token));
				}
			};

			_auth.checkAuthStatus = function () {
				var deferred = $q.defer();

				_auth.validateToken().then(function(response) {
					isAuthenticated = (response.status === 200);
					if(isAuthenticated) {
						$injector.invoke(['connectionTimeoutService', function(connectionTimeoutService) {
							connectionTimeoutService.run();
						}]);
					}
					deferred.resolve(isAuthenticated);
				}, function () {
					isAuthenticated = false;
					deferred.resolve(isAuthenticated);
				});

				return deferred.promise;
			};

			_auth.validateToken = function() {
				var deferred = $q.defer();
				var currentToken = _auth.getCurrentToken();

				resourceDirectory.fetch().then(
					function (directory) {
						_auth.setAuthorizationRequestHeader(currentToken);
						$http.get(directory['mhpuser']).then(function(response) {
							deferred.resolve(response);
						}, function(error) {
							deferred.resolve(error);
						});
					},
					function(error) {
						deferred.reject(error);
					}
				);

				return deferred.promise;
			};

			_auth.getCurrentToken = function () {
				var storedToken = sessionStorage.getItem('token');
				if (storedToken !== undefined && storedToken !== 'undefined' && storedToken !== null && storedToken !== 'null') {
					return JSON.parse(storedToken);
				}
				return null;
			};

			_auth.setAuthorizationRequestHeader = function(currentToken) {
				$http.defaults.headers.common.Authorization = "Bearer " + currentToken;
			};

			_auth.isAuthenticated = function () {
				return isAuthenticated;
			};

			_auth.deleteCookie = function (cookieName) {
				document.cookie = encodeURIComponent(cookieName) + '=deleted; expires=' + new Date(0).toUTCString();
			};

			_auth.deleteToken = function () {
				var deferred = $q.defer();

				resourceDirectory.fetch().then(
					function (directory) {
						$http.delete(directory['token'])
							.success(function (response) {
								deferred.resolve(response);
							})
							.error(function (error) {
								deferred.reject(error);
							});
					}, function (error) {
						deferred.reject(error);
					});

				return deferred.promise;
			};

			_auth.wipeSessionData = function () {
				isAuthenticated = false;
				var deferred = $q.defer();
				var storedToken = sessionStorage.getItem('token'),
					currentToken = '';
				if (storedToken !== undefined && storedToken !== null) {
					currentToken = JSON.parse(storedToken);
					_auth.deleteToken(currentToken).finally(function () {
						sessionStorage.removeItem('token');
						sessionStorage.removeItem('staffUserDisclaimerAccepted');
						$http.defaults.headers.common.Authorization = "";
						_auth.deleteCookie('JSESSIONID');
						deferred.resolve();
					});
				} else {
					deferred.resolve();
				}
				return deferred.promise;
			};

			_auth.logoutRedirectToLaunchpad = function () {
				_auth.wipeSessionData().then(function () {
					localResourceDirectoryService.fetch().then(function(directory) {
						var logoutUrl = directory['logout'] + '?' + new Date().getTime();
						$window.open(logoutUrl, "_self");
					});
				});
			};

			return _auth;
		});
});